(* ::Package:: *)

(* ::Title:: *)
(*Output routines*)


(* ::Text:: *)
(*12.02.09, CD - Fixed a bug in WriteLHAFile[]*)
(*02.11.10, NC - Fixed bug in ConvertLHAFile[] (Added InputForm to output.)*)
(*02.15.10, BF - Fixed bugs in ConvertLHAFile[] and WriteRestrictionFile[]. *)
(*06.10.10, NC - Improved ReadLHAFile.  Removed ConvertLHAFile.*)
(*11.02.12, NC - Fixed WriteParams.*)
(*17.01.13, NC - Fixed 1/0 problem.*)


(* ::Section::Closed:: *)
(*Screen output*)


PrintOutputCreaList[crealist_] := Block[{output},
    Do[Print["Particle ", kk, " : ", FieldType[crealist[[kk]]], " , ", crealist[[kk]]], {kk,Length[crealist]}];
    Print["Vertex:"]];
    
FieldType[field_] := "Scalar" /; ScalarFieldQ[field];
FieldType[field_] := "Vector" /; VectorFieldQ[field];
FieldType[field_] := "Dirac" /; DiracFieldQ[field];
FieldType[field_] := "Spin 2" /; Spin2FieldQ[field];
FieldType[field_] := "Majorana" /; MajoranaFieldQ[field];
FieldType[CC[field_]] := "Reversed Majorana" /; MajoranaFieldQ[field];
FieldType[CC[field_]] := "Reversed Dirac" /; DiracFieldQ[field];
FieldType[CCbar[field_]] := "Reversed Majorana" /; MajoranaFieldQ[field];
FieldType[CCbar[field_]] := "Reversed Dirac" /; DiracFieldQ[field];
FieldType[field_] := "Complex Spin 3/2" /; CSpin32FieldQ[field];
FieldType[field_] := "Real Spin 3/2" /; RSpin32FieldQ[field];
FieldType[CC[field_]] := "Reversed Real Spin 3/2" /; RSpin32FieldQ[field];
FieldType[CC[field_]] := "Reversed Complex Spin 3/2" /; CSpin32FieldQ[field];
FieldType[CCbar[field_]] := "Reversed Real Spin 3/2" /; RSpin32FieldQ[field];
FieldType[CCbar[field_]] := "Reversed Complex Spin 3/2" /; CSpin32FieldQ[field];

PrintScreenOutput[vl_] := Block[{psocrea, psovert},
      psocrea = vl[[All, 1, All, 1]];
      psovert = vl[[All, 2]];
       Do[Print["(* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *)"];
           Print["Vertex ", kk];
           PrintOutputCreaList[psocrea[[kk]]];
           Print[psovert[[kk]]],
           {kk, Length[vl]}];
       Print["(* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *)"]];    
       
      



(* ::Section:: *)
(*TeX output*)


(* ::Subsection::Closed:: *)
(*Useful functions*)


FeynRulesTeXForm[expr_]:=StringReplace[ToString[StandardForm[expr], TeXForm], {"\\left" -> "\\big","\\right" -> "\\big", "[" -> "(", "]" -> ")"}]

(* MW edit *)
(* TeXVertex[v] creates a string describing the vertex v in LaTeX code *)
TeXVertex[{f_,i_}]:=Block[{ppos,pstr},
  ppos = Position[PartList, {{_,f}, _}, 1];
  If[ppos==={},
    ppos = Position[PartList, {{_,anti[f]}, _}];
    If[ppos==={},
      pstr = FeynRulesTeXForm[f];
      ,
      pstr = PartList[[ppos[[1,1]], 2, 1, 12]];
    ];
  ,
    pstr = PartList[[ppos[[1,1]], 2, 1, 11]];
  ];
  
  "\\{"<>pstr<>", "<>ToString[i]<>"\\}"
];

TeXVertices[texvertlist_, of_] := Block[{texvtypel, texmax, texmin, texselect},
      texvtypel = Length /@ Table[texvertlist[[jj1, 1]], {jj1, Length[texvertlist]}];
      texmax = Max[texvtypel];
      texmin = Min[texvtypel];
      Do[texselect = Select[texvertlist, Length[#[[1]]] == jj2 &];
         WriteString[of, "\\subsection{ ", FeynRulesTeXForm[jj2], "-point vertices}\n"];
         WriteString[of, "\n"];
         WriteString[of, "\\begin{itemize}\n"];
         Do[WriteString[of, "\\item\n"];
            WriteString[of, "Vertex $", TeXVertex[texselect[[jj3, 1, 1]]], " $"]; (* MW edit: use TeXVertex instead of FeynRulesTeXForm here ... *)
            Do[WriteString[of, ", $", TeXVertex[texselect[[jj3, 1, jj4]]], " $"], {jj4, 2, jj2}]; (* MW edit: ...and here *)
            WriteString[of, "\n"];
            WriteString[of, "\\begin{respr}\n"];
            WriteString[of, FeynRulesTeXForm[Expand[texselect[[jj3, 2]]]]];
            WriteString[of, "\\end{respr}\n"],
         {jj3, Length[texselect]}];
         WriteString[of, "\\end{itemize}\n"];
         WriteString[of, "\n"],
         {jj2, texmin, texmax}]];



FRMakeTeXOut[outfile_, vl_] := Block[{texclass, texindlist},
      texindlist = MR$IndexList /. Index -> Identity;
      OpenWrite[outfile];
      WriteString[outfile, "%\n"];
      WriteString[outfile, "%\n"];
      WriteString[outfile, "% This TeX-file has been automatcally generated by FeynRules.\n"];
      WriteString[outfile, "%\n"];
      WriteString[outfile, "%\n"];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\\documentclass[11pt]{article}\n"];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\\usepackage{amsfonts}\n"];
      WriteString[outfile, "\\usepackage{amsmath}\n"];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\\newenvironment{respr}[0]{\\sloppy\\begin{flushleft}\\hspace*{0.75cm}\\(}{\\)\\end{flushleft}\\fussy}\n"];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\\setlength{\\topmargin}{-.2 cm}\n"];
      WriteString[outfile, "\\setlength{\\evensidemargin}{.0 cm}\n"];
      WriteString[outfile, "\\setlength{\\oddsidemargin}{.0 cm}\n"];
      WriteString[outfile, "\\setlength{\\textheight}{8.5 in}\n"];
      WriteString[outfile, "\\setlength{\\textwidth}{6.4 in}\n"];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\\begin{document}\n"];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\\section{Model description}\n"];
      WriteString[outfile, "This file contains the Feynman rules for the model \\verb+", ToString[M$ModelName], "+.\n"];
      WriteString[outfile, "The Feynman rules have been generated automatically by FeynRules", FR$VersionNumber, ".\n"];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\\subsection{Model information}\n"];
      WriteString[outfile, "\n"];
      WriteString[outfile, "Author(s) of the model file: \\\\\n"];
      If[(MR$Authors === {MR$Null}) || Not[ValueQ[MR$Authors]], WriteString[outfile, "\\indent N / A\\\\\n"],
         Do[ WriteString[outfile, "\\indent ", MR$Authors[[kk]], "\\\\\n"], {kk,Length[MR$Authors]}]];
      If[(MR$Institutions =!= {MR$Null}) && ValueQ[MR$Institutions], 
         WriteString[outfile, "Institution(s):\\\\\n"];
         Do[ WriteString[outfile, "\\indent ", MR$Institutions[[kk]], "\\\\\n"], {kk,Length[MR$Institutions]}]];
      If[(MR$Emails =!= {MR$Null}) && ValueQ[MR$Emails],
         WriteString[outfile, "Emails:\\\\\n"];
         Do[ WriteString[outfile, "\\indent ", MR$Emails[[kk]], "\\\\\n"], {kk,Length[MR$Emails]}]];
      If[(MR$Date =!= {MR$Null}) && ValueQ[MR$Date],WriteString[outfile, "Date: ", MR$Date, "\\\\\n"]];
      If[(MR$References =!= {MR$Null}) && ValueQ[MR$References],
         WriteString[outfile, "References used to build the model file:\\\\\n"];
         Do[ WriteString[outfile, "\\indent ", MR$References[[kk]], "\\\\\n"], {kk,Length[MR$References]}]];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\\subsection{Index description}\n"];
      WriteString[outfile, "\n"];
      If[texindlist === {},
         WriteString[outfile, "No Indices defined.\n"],
         WriteString[outfile, "\\begin{center}"];
         WriteString[outfile, "\\begin{tabular}{|c|c|c|}\n"];
         WriteString[outfile, "\\hline\n"];
         WriteString[outfile, "Index & Index range & Symbol"];
         Do[WriteString[outfile, "\\\\ \n"];
            WriteString[outfile, "\\hline\n"];
            WriteString[outfile, ToString[texindlist[[kk]]], " & "];
            WriteString[outfile, 
            Which[Length[MRIndexRange[Index[texindlist[[kk]]]]] == 1, 
                       ToString @@ MRIndexRange[Index[texindlist[[kk]]]],
                  Length[MRIndexRange[Index[texindlist[[kk]]]]] == 2, 
                       StringJoin[ToString[MRIndexRange[Index[texindlist[[kk]]]][[1]]], " , ", ToString[MRIndexRange[Index[texindlist[[kk]]]][[2]]]],
                  Head[MRIndexRange[Index[texindlist[[kk]]]]] === MRIndexRange,
                       "N/A",
                  True, 
                       StringJoin[ToString[First[MRIndexRange[Index[texindlist[[kk]]]]]], " \\ldots ", ToString[Last[MRIndexRange[Index[texindlist[[kk]]]]]]]]];
            WriteString[outfile, " & ", If[Head[ MR$IndForm[texindlist[[kk]]]] === MR$IndForm, "N/A", StringJoin["$ ", ToString[MR$IndForm[texindlist[[kk]]]], " $"]]],
            {kk, 1, Length[texindlist]}];
         WriteString[outfile, "\n\\\\ \\hline\n"];
         WriteString[outfile, "\\end{tabular}"];
         WriteString[outfile, "\\end{center}"];];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\\subsection{Particle content of the model}\n"];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\\begin{enumerate}\n"];
      Do[texclass = MR$ClassNameList[[kk]];
        texclassind = $IndList[texclass] /. Index -> Identity;
        WriteString[outfile, "\\item \n"];
        WriteString[outfile, "\\begin{tabular}{ll}\n"];
        WriteString[outfile, "Class: ", ToString[MR$Class[texclass] /. xxx_[_] :> xxx ] <> "(" <> ToString[MR$Class[texclass] /. _[xxx_] :> xxx ] <> ")" , " = $", TeXCN[texclass], "$, &  Fieldtype: ", MR$FieldType[texclass]];
        WriteString[outfile, If[Unphysical /.  MR$ClassesRules[MR$Class[texclass]] /. {Unphysical -> False}, " (Unphysical).\\\\ \n", ".\\\\ \n"]];
        WriteString[outfile, "\\multicolumn{2}{l}{Indices: "];
        If[texclassind === {}, WriteString[outfile, "N/A.}\\\\ \n"],
           WriteString[outfile, texclassind[[1]]];
           Do[WriteString[outfile, ", ", texclassind[[ll]]], {ll, 2, Length[texclassind]}];
           WriteString[outfile, ".}\\\\ \n"]];
        If[MR$ClassMembers[texclass] =!= {texclass},
          WriteString[outfile, "\\multicolumn{2}{l}{Class Members: ", TeXForm[MR$ClassMembers[texclass][[1]]]];
          If[Length[MR$ClassMembers[texclass]] != 1,
             Do[WriteString[outfile, ", ", MR$ClassMembers[texclass][[ll]]], {ll, 2, Length[MR$ClassMembers[texclass]]}]];
             WriteString[outfile, ".}\n"]];
        WriteString[outfile, "\\end{tabular}\n"],
        {kk, Length[MR$ClassNameList]}];
      WriteString[outfile, "\\end{enumerate}\n"];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\\section{Vertices}\n"];
      WriteString[outfile, "\n"];
      TeXVertices[vl, outfile];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\\end{document}\n"];
      Print["TeX output written on ", outfile, "."];
      Close[outfile];];


(* ::Subsection:: *)
(*WriteTeXOutput*)


Options[WriteTeXOutput] = {PrintLagrangian -> False, Output -> Automatic};

WriteTeXOutput[argx__]:=WriteTeXOutputList[{argx}] /; And @@ ((Head[#] =!= Rule &) /@ {argx});

WriteTeXOutput[argx__]:=WriteTeXOutputList[Select[{argx}, Head[#] =!= Rule &], Sequence @@ Select[{argx}, Head[#] === Rule &]] /; Not[And @@ ((Head[#] =!= Rule &) /@ {argx})];

WriteTeXOutputList[lagpieces_List, options___] := Block[{texclass, texindlist, texvl, outfile, tempoutfile, laglists, templaglists = {}},
      (*Setting the output file name *)
      tempoutfile = Output /. {options} /. Options[WriteTeXOutput];
      If[tempoutfile =!= Automatic,
         If[Not[StringQ[tempoutfile]], tempoutfile = ToString[tempoutfile]];
         outfile = If[StringLength[tempoutfile]<6, StringJoin[tempoutfile, ".tex"],
                      If[(StringTake[tempoutfile,-4] =!= ".tex"), StringJoin[tempoutfile, ".tex"], tempoutfile]],
         (*Default value *)
         outfile = StringJoin[If[StringQ[M$ModelName], M$ModelName, ToString[M$ModelName]], ".tex"]];
      texindlist = MR$IndexList /. Index -> Identity;
      OpenWrite[outfile];
      WriteString[outfile, "%\n"];
      WriteString[outfile, "%\n"];
      WriteString[outfile, "% This TeX-file has been automatically generated by FeynRules.\n"];
      WriteString[outfile, "%\n"];
      WriteString[outfile, "% C. Duhr, 2008\n"];
      WriteString[outfile, "%\n"];
      WriteString[outfile, "%\n"];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\\documentclass[11pt]{article}\n"];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\\usepackage{amsfonts}\n"];
      WriteString[outfile, "\\usepackage{amsmath}\n"];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\\newenvironment{respr}[0]{\\sloppy\\begin{flushleft}\\hspace*{0.75cm}\\(}{\\)\\end{flushleft}\\fussy}\n"];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\\setlength{\\topmargin}{-.2 cm}\n"];
      WriteString[outfile, "\\setlength{\\evensidemargin}{.0 cm}\n"];
      WriteString[outfile, "\\setlength{\\oddsidemargin}{.0 cm}\n"];
      WriteString[outfile, "\\setlength{\\textheight}{8.5 in}\n"];
      WriteString[outfile, "\\setlength{\\textwidth}{6.4 in}\n"];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\\begin{document}\n"];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\\section{Model description}\n"];
      WriteString[outfile, "This file contains the Feynman rules for the model \\verb+", ToString[M$ModelName], "+.\n"];
      WriteString[outfile, "The Feynman rules have been generated automatically by FeynRules", FR$VersionNumber, ".\n"];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\\subsection{Model information}\n"];
      WriteString[outfile, "\n"];
      WriteString[outfile, "Author(s) of the model file: \\\\\n"];
      If[(MR$Authors === {MR$Null}) || Not[ValueQ[MR$Authors]], WriteString[outfile, "\\indent N / A\\\\\n"],
         Do[ WriteString[outfile, "\\indent ", MR$Authors[[kk]], "\\\\\n"], {kk,Length[MR$Authors]}]];
      If[(MR$Institutions =!= {MR$Null}) && ValueQ[MR$Institutions], 
         WriteString[outfile, "Institution(s):\\\\\n"];
         Do[ WriteString[outfile, "\\indent ", MR$Institutions[[kk]], "\\\\\n"], {kk,Length[MR$Institutions]}]];
      If[(MR$Emails =!= {MR$Null}) && ValueQ[MR$Emails],
         WriteString[outfile, "Emails:\\\\\n"];
         Do[ WriteString[outfile, "\\indent ", MR$Emails[[kk]], "\\\\\n"], {kk,Length[MR$Emails]}]];
      If[(MR$Date =!= {MR$Null}) && ValueQ[MR$Date],WriteString[outfile, "Date: ", MR$Date, "\\\\\n"]];
      If[(MR$References =!= {MR$Null}) && ValueQ[MR$References],
         WriteString[outfile, "References used to build the model file:\\\\\n"];
         Do[ WriteString[outfile, "\\indent ", MR$References[[kk]], "\\\\\n"], {kk,Length[MR$References]}]];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\\subsection{Index description}\n"];
      WriteString[outfile, "\n"];
      If[texindlist === {},
         WriteString[outfile, "No Indices defined.\n"],
         WriteString[outfile, "\\begin{center}"];
         WriteString[outfile, "\\begin{tabular}{|c|c|c|}\n"];
         WriteString[outfile, "\\hline\n"];
         WriteString[outfile, "Index & Index range & Symbol"];
         Do[WriteString[outfile, "\\\\ \n"];
            WriteString[outfile, "\\hline\n"];
            WriteString[outfile, ToString[texindlist[[kk]]], " & "];
            WriteString[outfile, 
            Which[Length[MRIndexRange[Index[texindlist[[kk]]]]] == 1, 
                       ToString @@ MRIndexRange[Index[texindlist[[kk]]]],
                  Length[MRIndexRange[Index[texindlist[[kk]]]]] == 2, 
                       StringJoin[ToString[MRIndexRange[Index[texindlist[[kk]]]][[1]]], " , ", ToString[MRIndexRange[Index[texindlist[[kk]]]][[2]]]],
                  Head[MRIndexRange[Index[texindlist[[kk]]]]] === MRIndexRange,
                       "N/A",
                  True, 
                       StringJoin[ToString[First[MRIndexRange[Index[texindlist[[kk]]]]]], " \\ldots ", ToString[Last[MRIndexRange[Index[texindlist[[kk]]]]]]]]];
            WriteString[outfile, " & ", If[Head[ MR$IndForm[texindlist[[kk]]]] === MR$IndForm, "N/A", StringJoin["$ ", ToString[MR$IndForm[texindlist[[kk]]]], " $"]]],
            {kk, 1, Length[texindlist]}];
         WriteString[outfile, "\n\\\\ \\hline\n"];
         WriteString[outfile, "\\end{tabular}"];
         WriteString[outfile, "\\end{center}"];];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\\subsection{Particle content of the model}\n"];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\\begin{enumerate}\n"];
      Do[texclass = MR$ClassNameList[[kk]];
        texclassind = $IndList[texclass] /. Index -> Identity;
        WriteString[outfile, "\\item \n"];
        WriteString[outfile, "\\begin{tabular}{ll}\n"];
        WriteString[outfile, "Class: ", ToString[MR$Class[texclass] /. xxx_[_] :> xxx ] <> "(" <> ToString[MR$Class[texclass] /. _[xxx_] :> xxx ] <> ")", " = $ ", TeXCN[texclass], " $, &  Fieldtype: ", MR$FieldType[texclass]];
        WriteString[outfile, If[Unphysical /.  MR$ClassesRules[MR$Class[texclass]] /. {Unphysical -> False}, " (Unphysical).\\\\ \n", ".\\\\ \n"]];
        WriteString[outfile, "\\multicolumn{2}{l}{Indices: "];
        If[texclassind === {}, WriteString[outfile, "N/A.}\\\\ \n"],
           WriteString[outfile, texclassind[[1]]];
           Do[WriteString[outfile, ", ", texclassind[[ll]]], {ll, 2, Length[texclassind]}];
           WriteString[outfile, ".}\\\\ \n"]];
        If[MR$ClassMembers[texclass] =!= {texclass},
          WriteString[outfile, "\\multicolumn{2}{l}{Class Members: ", TeXForm[MR$ClassMembers[texclass][[1]]]];
          If[Length[MR$ClassMembers[texclass]] != 1,
             Do[WriteString[outfile, ", ", MR$ClassMembers[texclass][[ll]]], {ll, 2, Length[MR$ClassMembers[texclass]]}]];
             WriteString[outfile, ".}\n"]];
        WriteString[outfile, "\\end{tabular}\n"],
        {kk, Length[MR$ClassNameList]}];
      WriteString[outfile, "\\end{enumerate}\n"];
      WriteString[outfile, "\n"];
      laglists = PrintLagrangian /. {options} /. Options[WriteTeXOutput];
      laglists = Flatten[{laglists}];
      If[laglists =!= {False},
         (* Check if all lagrnagians have a value *)
         Do[If[Not[ValueQ[laglists[[kk]]]], Print["Lagrangian at position ", ToString[kk], " has no value and will be ignored."], 
               templaglists = Append[templaglists, laglists[[kk]]]], 
            {kk, Length[laglists]}];
         laglists = templaglists;
         (* Start reading out *)
         WriteString[outfile, "\n"];
         WriteString[outfile, "%%\n"];   
         WriteString[outfile, "%% The Lagrangian\n"];
         WriteString[outfile, "%%\n"];
         WriteString[outfile, "\n"];
         WriteString[outfile, "\\section{The lagrangian}\n"];
         WriteString[outfile, "\n"];
         Do[WriteString[outfile, "\n"];
            WriteString[outfile, "%\n"];   
            WriteString[outfile, "% Lagrangian ", ToString[kk], "\n"];
            WriteString[outfile, "%\n"];
            WriteString[outfile, "\n"];
            WriteString[outfile, "\\noindent Lagrangian ", ToString[kk], "\n"];
            WriteString[outfile, "\n"];
            WriteString[outfile, "\\begin{respr}\n"];
            WriteString[outfile, FeynRulesTeXForm[Expand[laglists[[kk]] /. Dot -> FR$Dot /. FR$Dot -> Dot]]];
            WriteString[outfile, "\\end{respr}\n"],
            {kk, Length[laglists]}]];
      WriteString[outfile, "\n"];
      WriteString[outfile, "%%\n"];   
      WriteString[outfile, "%% The Vertices\n"];
      WriteString[outfile, "%%\n"];
      WriteString[outfile, "\\section{Vertices}\n"];
      WriteString[outfile, "\n"];
(*      Which[Length[lagpieces] == 0,
                  Message[TeX::NoVert],
            Length[lagpieces] == 1,
                  TeXVertices[Vertices @@ lagpieces, outfile],
            True,
                  TeXVertices[Join @@ (Vertices /@ lagpieces), outfile]];*)
      Which[Length[lagpieces] == 0,
                  Message[TeX::NoVert],
            Length[lagpieces] == 1,
                  TeXVertices[Identity @@ lagpieces, outfile],
            True,
                  TeXVertices[MergeVertices @@ lagpieces, outfile]];
      WriteString[outfile, "\n"];
      WriteString[outfile, "\\end{document}\n"];
      Print["TeX output written on ", outfile, "."];
      Close[outfile];];


(* ::Section::Closed:: *)
(*ParamFile Output/Input*)


(* ::Subsection::Closed:: *)
(*CreateParamLists[]*)


CreateParamLists[NV_]:=Module[{j},
M$ExtParams={};
M$Masses={};
M$Widths={};
M$IntParams={};
M$IntParamsNumericalValues={};
Do[
If[ParamList[[j,2]]===Ext,
AppendTo[M$ExtParams,ParamList[[j,1]]->ParamList[[j,3]]/.NoValue[1]->1]];
If[ParamList[[j,2]]===Int,
AppendTo[M$IntParams,ParamList[[j,1]]->ParamList[[j,3]]]];
,{j,1,Length[ParamList]}];
Do[
If[MassList[[2,j,3]]=!=Internal,
AppendTo[M$Masses,MassList[[2,j,2]]->MassList[[2,j,3]]/.NoValue[1]->1]];
,{j,1,Length[MassList[[2]]]}];
Do[
If[WidthList[[2,j,3]]=!=Internal,
AppendTo[M$Widths,WidthList[[2,j,2]]->WidthList[[2,j,3]]/.NoValue[1]->1]];
,{j,1,Length[WidthList[[2]]]}];
M$IntParamsNumericalValues=M$IntParams;
If[NV,Do[
  M$IntParamsNumericalValues[[j,2]]=NumericalValue[M$IntParamsNumericalValues[[j,2]](*//.M$IntParams/.M$ExtParams/.M$Masses/.M$Widths*)];
,{j,1,Length[M$IntParams]}]];
];


(* ::Subsection::Closed:: *)
(*SetNumericalValues[]*)


SetNumericalValues[]:=Module[{tmp,j,ReverseParamRules,ReverseParamRulesSimplified},
ReverseParamRules=ParamRules/.(a_->b_)->(b->a);
ReverseParamRulesSimplified=ReverseParamRules/.Index[_,a_]->a;
CreateParamLists[False];
Do[
NumericalValue[MassList[[2,j,2]]]=N[MassList[[2,j,2]]//.M$IntParams/.M$ExtParams/.M$Masses];
,{j,1,Length[MassList[[2]]]}];
Do[
NumericalValue[WidthList[[2,j,2]]]=N[WidthList[[2,j,2]]//.M$IntParams/.M$ExtParams/.M$Widths];
,{j,1,Length[WidthList[[2]]]}];
Do[
NumericalValue[ParamList[[j,1]]]=N[ParamList[[j,1]]//.M$IntParams/.M$ExtParams/.M$Masses/.M$Widths];
NumericalValue[ParamList[[j,1]]/.ReverseParamRules]=N[ParamList[[j,1]]//.M$IntParams/.M$ExtParams/.M$Masses/.M$Widths];
NumericalValue[ParamList[[j,1]]/.ReverseParamRulesSimplified]=N[ParamList[[j,1]]//.M$IntParams/.M$ExtParams/.M$Masses/.M$Widths];
,{j,1,Length[ParamList]}];
Do[
  M$IntParamsNumericalValues[[j,2]]=NumericalValue[M$IntParamsNumericalValues[[j,2]](*//.M$IntParams/.M$ExtParams/.M$Masses/.M$Widths*)];
,{j,1,Length[M$IntParams]}];

]


(* ::Subsection::Closed:: *)
(*UpdateParameters[options]*)


UpdateParameters[options___]:=Module[{j,opts,ParamRulesSimplified},
(*If[Head[options]===List,
opts=options/.ParamRules;,*)
ParamRulesSimplified=ParamRules/.Index[_,a_]->a;
opts=Flatten[{options}]/.ParamRules/.ParamRulesSimplified;
(*];*)
(*ParamList*)
Do[
If[ParamList[[j,2]]===Ext,ParamList[[j,3]]=ParamList[[j,1]]/.opts/.ParamList[[j,1]]->ParamList[[j,3]]];
,{j,1,Length[ParamList]}];
(*EParamList*)
Do[
EParamList[[j,2,i,2,-3]]=EParamList[[j,2,i,2,1]]/.opts/.EParamList[[j,2,i,2,1]]->EParamList[[j,2,i,2,-3]];
,{j,1,Length[EParamList]},{i,1,Length[EParamList[[j,2]]]}];
(*MassList*)
Do[
If[MassList[[2,j,3]]=!=Internal,MassList[[2,j,3]]=MassList[[2,j,2]]/.opts/.MassList[[2,j,2]]->MassList[[2,j,3]]];
,{j,1,Length[MassList[[2]]]}];
(*WidthList*)
Do[
If[WidthList[[2,j,3]]=!=Internal,WidthList[[2,j,3]]=WidthList[[2,j,2]]/.opts/.WidthList[[2,j,2]]->WidthList[[2,j,3]]];
,{j,1,Length[WidthList[[2]]]}];
SetNumericalValues[];
If[Global`FR$Parallelize,
   UpdateFRDistributedVariables[]
     ];

];


(* ::Subsection::Closed:: *)
(*WriteParameters[options]*)


Options[WriteParameters] = {Output :> M$ModelName <> ".pars", ScreenOutput -> True, NumericalOnly -> False} 


(* ::Subsection::Closed:: *)
(*MakeNumericalOnly*)


(* ::Text:: *)
(*This function cleans up the parameters lists, and removes certain non numeric entries, e.g.,*)
(**)
(*   *   NoValue[1]*)
(*   *  Internal*)
(*   *  Double entries*)


MakeNumericalOnly[list_List] := Union[DeleteCases[list, Rule[_, NoValue[_]]|Rule[_,Internal]]];


(* ::Subsection::Closed:: *)
(*WriteParameters*)


WriteParameters[options___]:=Module[{file,fileName,j,comma=False,ReverseParamRules},
fileName = Output /. {options} /. Options[WriteParameters];
ReverseParamRules=ParamRules/.(a_->b_)->(b->a);

(* Create the list M$ExtParams, etc. *)
CreateParamLists[True];

(* Clean them oup if necessary *)
If[NumericalOnly /. {options} /. Options[WriteParameters],
   M$ExtParams = MakeNumericalOnly[M$ExtParams]; 
   M$IntParams = MakeNumericalOnly[M$IntParams]; 
   M$Masses = MakeNumericalOnly[M$Masses]; 
   M$Widths = MakeNumericalOnly[M$Widths];
  ];




If[ScreenOutput /. {options} /. Options[WriteParameters], Print["Writing output to "<>fileName<>"."]];
file=OpenWrite[fileName];

WriteString[file,"(***********************************************)\n"];
WriteString[file,"(*   Parameter file for "<>M$ModelName<>".        *)\n"];
WriteString[file,"(*   Automatically generated by FeynRules.     *)\n"];
WriteString[file,"(***********************************************)\n"];

(*External parameters*)
WriteString[file,"(***********************************************)\n"];
WriteString[file,"(*             External Parameters             *)\n"];
If[Not[FR$FeynArtsInterface],
   WriteString[file,"(*    The numerical values can be changed.     *)\n"];
  ];
WriteString[file,"(***********************************************)\n"];
WriteString[file,"\nM$ExtParams = {\n"];
comma=False;
Do[
If[comma&&ParamList[[j,2]]===Ext,WriteString[file,",\n"]];
If[ParamList[[j,2]]===Ext,
comma=True;
(*WriteString[file,"\t"<>ToString[InputForm[ParamList[[j,1]]]]<>" -> "<>ToString[N[ParamList[[j,3]]]]];*)
WriteString[file,"\t"<>ToString[InputForm[M$ExtParams[[j]]]]];
];
(*,{j,1,Length[ParamList]}];*)
,{j,1,Length[M$ExtParams]}];
WriteString[file,"\n};\n\n"];

(*Masses*)
WriteString[file,"(***********************************************)\n"];
WriteString[file,"(*                  Masses                     *)\n"];
If[Not[FR$FeynArtsInterface],
   WriteString[file,"(*    The numerical values can be changed.     *)\n"];
  ];
WriteString[file,"(***********************************************)\n"];
WriteString[file,"\nM$Masses = {\n"];
comma=False;
Do[
If[comma,WriteString[file,",\n"]];
comma=True;
(*WriteString[file,"\t"<>ToString[InputForm[MassList[[2,j,2]]]]<>" -> "<>ToString[NumericalValue[MassList[[2,j,2]]]]];
,{j,1,Length[MassList[[2]]]}];*)
WriteString[file,"\t"<>ToString[InputForm[M$Masses[[j]]]]];
,{j,1,Length[M$Masses]}];
WriteString[file,"\n};\n\n"];

(*Widths*)
WriteString[file,"(***********************************************)\n"];
WriteString[file,"(*                  Widths                     *)\n"];
If[Not[FR$FeynArtsInterface],
   WriteString[file,"(*    The numerical values can be changed.     *)\n"];
  ];
WriteString[file,"(***********************************************)\n"];
WriteString[file,"\nM$Widths = {\n"];
comma=False;
Do[
If[comma,WriteString[file,",\n"]];
comma=True;
(*WriteString[file,"\t"<>ToString[InputForm[WidthList[[2,j,2]]]]<>" -> "<>ToString[NumericalValue[WidthList[[2,j,2]]]]];
,{j,1,Length[WidthList[[2]]]}];*)
WriteString[file,"\t"<>ToString[InputForm[M$Widths[[j]]]]];
,{j,1,Length[M$Widths]}];
WriteString[file,"\n};\n\n"];

If[Not[FR$FeynArtsInterface],
   (*Internal parameters numerical values*)
   WriteString[file,"(***********************************************)\n"];
   WriteString[file,"(*             Internal Parameters             *)\n"];
   WriteString[file,"(*     The numerical values are ignored.       *)\n"];
   WriteString[file,"(*     They are just here for reference.       *)\n"];
   WriteString[file,"(***********************************************)\n"];
   WriteString[file,"M$IntParamsNumericalValues = {\n"];
   comma=False;
   Do[
      If[comma&&ParamList[[j,2]]===Int,WriteString[file,",\n"]];
      If[ParamList[[j,2]]===Int,
      comma=True;
      WriteString[file,"\t"<>ToString[InputForm[ParamList[[j,1]]]]<>" -> "<>ToString[InputForm[NumericalValue[ParamList[[j,1]]]]]];
      ];
    ,{j,1,Length[ParamList]}];
    WriteString[file,"\n};\n\n"];
];

(*Internal parameters functions*)
WriteString[file,"(***********************************************)\n"];
WriteString[file,"(*             Internal Parameters             *)\n"];
If[Not[FR$FeynArtsInterface],
   WriteString[file,"(*       Changes are currently ignored.        *)\n"];
   WriteString[file,"(*     They are just here for reference.       *)\n"];
   ];
WriteString[file,"(***********************************************)\n"];
WriteString[file,"\nM$IntParams = {\n"];
comma=False;
Do[
If[comma,WriteString[file,",\n"]];
comma=True;
If[NumericalOnly /. {options} /. Options[WriteParameters],
	WriteString[file,"\t"<>ToString[InputForm[M$IntParamsNumericalValues[[j]]]]];,
	WriteString[file,"\t"<>ToString[InputForm[M$IntParams[[j]]]]];
];
,{j,1,Length[M$IntParams]}];
WriteString[file,"\n};\n\n"];

Close[file];
];


(* ::Subsection:: *)
(*ReadParameters[options]*)


ReadParameters[options___]:=Module[{file,fileName,j,comma=False,replList},
fileName=Input/.{options}/.Input->M$ModelName<>".pars";
If[ScreenOutput /.{options}/. ScreenOutput -> True, Print["Reading input from "<>fileName<>"."]];
Get[fileName];
replList=Flatten[{M$ExtParams,M$Masses,M$Widths}]/.List[a__]->a;
UpdateParameters[replList];
];


(* ::Section::Closed:: *)
(*WriteRestrictionFile*)


(* ::Text:: *)
(*This routine writes a restriction file, which puts to zero the external parameters with zero numerical value.*)


Options[WriteRestrictionFile]={Output->"ZeroValues.rst"};


WriteRestrictionFile[options___]:=Block[{templist, itemplist,mtemplist,wtemplist, outfile,symbrules = Reverse/@ParamRules, MyRule},
   outfile=Output/.{options}/.Options[WriteRestrictionFile];
   WriteParameters[Output->"FR_Template.pars",ScreenOutput->False];
   DeleteFile["FR_Template.pars"];

   templist=M$ExtParams/.{Rule|RuleDelayed:>MyRule}/.symbrules/.Index[_,k_]:>k;
   templist=Cases[templist,MyRule[_,0.|0]]/.MyRule[a_, b_]:>Rule[a, 0];


   itemplist = M$IntParams /. {Rule|RuleDelayed:>MyRule}/.symbrules/.Index[_,k_]:>k;
   itemplist = Cases[itemplist, MyRule[_, _?(NumericalValue[#] == 0. &)]];
   itemplist = itemplist /. MyRule[par_, _] :> MyRule[par, 0] /. MyRule :> Rule;

   mtemplist = M$Masses /. {Rule|RuleDelayed:>MyRule}/.symbrules/.Index[_,k_]:>k;
   mtemplist = Cases[mtemplist, MyRule[_, _?(NumericalValue[#] == 0. &)]];
   mtemplist = mtemplist /. MyRule[par_, _] :> MyRule[par, 0] /. MyRule :> Rule;

   wtemplist = M$Widths /. {Rule|RuleDelayed:>MyRule}/.symbrules/.Index[_,k_]:>k;
   wtemplist = Cases[wtemplist, MyRule[_, _?(NumericalValue[#] == 0. &)]];
   wtemplist = wtemplist /. MyRule[par_, _] :> MyRule[par, 0] /. MyRule :> Rule;


   templist = Join[templist, itemplist, mtemplist, wtemplist];

   If[Length[templist] == 0, Print["All external parameters have a non zero numerical value. No restriction file written."],
   (* else *)
      OpenWrite[outfile];
      WriteString[outfile,"(********************************************************************)\n"];
      WriteString[outfile,"(*                                                                *)\n"];
      WriteString[outfile,"(*                                                                *)\n"];
      WriteString[outfile,"(*       This model restriction removes all external parameters   *)\n"];
      WriteString[outfile,"(*           with zero numerical value.                           *)\n"];
      WriteString[outfile,"(*                                                                *)\n"];
      WriteString[outfile,"(*       This file was automatically generated by FeynRules       *)\n"];
      WriteString[outfile,"(*                                                                *)\n"];
      WriteString[outfile,"(********************************************************************)\n"];
      WriteString[outfile,"\n"];
      WriteString[outfile,"\n"];
      WriteString[outfile,"     M$Restrictions = {\n"];
      WriteString[outfile,"         "<>ToString[#,InputForm]<>",\n"]&/@Most[templist];
      WriteString[outfile,"         "<>ToString[Last[templist],InputForm]<>"\n"];
      WriteString[outfile,"     }"];
      Close[outfile];
 
      Print["Model restriction written on ",outfile]]];


(* ::Section:: *)
(*LHA file Output/Input*)


(* ::Subsection::Closed:: *)
(*WriteQNumbers*)


WriteQNumbers[outfile_]:=Block[{bsmpart},
(* Write the QNUMBERS LH block to an open output stream *)
(* Note: Only physical states are written, auxiliary fields like T1 (8000001) are ignored. *)
(*Remove SM particles *)
   bsmpart = DeleteCases[Cases[Join@@PartList[[All,2]],{__,NoGS}]/.NoPDG->Identity,{_,_,U,__}|_?(MemberQ[FR$SMPDGs,#[[9]]]&)];
(* Only do it if bsmparts is not empty *)
   If[bsmpart =!= {},
      WriteString[outfile,"\n"];
      WriteString[outfile,"#===========================================================\n"];
      WriteString[outfile,"# QUANTUM NUMBERS OF NEW STATE(S) (NON SM PDG CODE) IF ANY\n"];
      WriteString[outfile,"#===========================================================\n"];
      WriteString[outfile,"\n"];
      Do[WriteString[outfile,"Block QNUMBERS ",ToString[ bsmpart[[num,9]]],"  # ",  bsmpart[[num,1]],"\n"];
         WriteString[outfile,"        1 ",ToString[3*Q[PartSymbol[bsmpart[[num,1]]]]],"  # 3 times electric charge\n"];
         WriteString[outfile,"        2 ",
                     Which[bsmpart[[num,3]]===S, "1",
                           bsmpart[[num,3]]===F, "2",
                           bsmpart[[num,3]]===V, "3",
                           bsmpart[[num,3]]===T, "5",
                           True,"1"],"  # number of spin states (2S+1)\n"];
         WriteString[outfile,"        3 ",
                     Which[bsmpart[[num,7]]===S, "1",
                           bsmpart[[num,7]]===T, "3",
                           bsmpart[[num,7]]===O, "8",
                           True,"1"],"  # colour rep (1: singlet, 3: triplet, 8: octet)\n"];
         WriteString[outfile,"        4 ",If[bsmpart[[num,1]]===bsmpart[[num,2]], "0", "1"],"  # Particle/Antiparticle distinction (0=own anti)\n"],
      {num,Length[bsmpart]}]]]


(* ::Subsection::Closed:: *)
(*WriteLHAFile[]*)


(* Based on the routine for the MG param_card by M. Herquet *)

Options[WriteLHAFile] = {Output -> "LHAFile.dat"};

WriteLHAFile[options___] := Block[{temp, outfile, WriteBlock, WriteBlockMass, WriteBlockDecay,  goldsnghosts, 
           tempMGdoublemasses, tempMGsinglemasses, tmplist, neweparams ={}, GSList, GSMasses, GSDecays, pdgtowidth, pdgtomass, parttemplist,
  DecomposeEParam, ToFortranNumber, MixMatList},
   outfile = Output /. {options} /. Options[WriteLHAFile];
   OpenWrite[outfile];
   WriteString[outfile, "#This file has been generated automatically by FeynRules\n"];

(* Initialisation *)

Clear[MGEParamList, MGIParamList, MGParList, MGMassList, MGWidthList, FRMGDefault];

(* The subroutines defining the printing of the different Blocks*)
    WriteBlock[xx_,file_]:=Module[{i},
       WriteString[file,"Block "<>ToString[xx[[1]]]<>"\n"];
       WriteString[file,"  "<>StringJoin[ToString[#]<>" "&/@#[[1]]]
          <>" "<>ToFortranNumber[#[[2]][[-2]]]<>"  # "<>ToString[#[[2,1]]]<>"\n"]&/@ xx[[2]];
    ];
    WriteBlockMass[xx_,file_]:=Module[{i},
       WriteString[file,"Block "<>ToString[xx[[1]]]<>"\n"];
       WriteString[file,"  "<>StringJoin[ToString[Abs[#]]<>" "&/@#[[1]]]
          <>" "<>ToFortranNumber[#[[3]]]<>"  # "<>ToString[#[[2]]]<>"\n"]&/@ xx[[2]];
    ];
    WriteBlockDecay[xx_,file_]:=Module[{i},
       WriteString[file,"DECAY  "<>StringJoin[ToString[Abs[#]]<>" "&/@#[[1]]]
          <>" "<>ToFortranNumber[#[[3]]]<>"  # "<>ToString[#[[2]]]<>"\n"]&/@ xx[[2]];
    ];

DecomposeEParam[param_] := Which[Length[param]==4, MapAt[NumericalValue,param,2],  
         Length[param]==6, MapAt[NumericalValue,param,4],
         True, param];

ToFortranNumber[xxxx_] := Block[{outputo, tempo},
     If[N[xxxx] == 0., outputo = "0.00000000E+00",
   (* else *)
     tempo = SetPrecision[MantissaExponent[xxxx], 15];
     tempo = {10*tempo[[1]], Rationalize[tempo[[2]] - 1]};
     tempo = {StringTake[StringReplace[ToString[N[Rationalize[tempo[[1]]], 15]],{"`9." -> ""}], 15], ToString[PaddedForm[tempo[[2]], 2, NumberPadding -> {"0", "0"}, SignPadding -> True, NumberSigns -> {"-", "+"}]]};
     outputo = tempo[[1]] <> "E" <> tempo[[2]]];
   outputo];

    (*                                                          *)
    (*            Preparing the particle and parameter input    *)
    (*                                                          *)

    Print["Checking parameter list..."];                                           

(*To get a nices output, we first remove the ghosts and goldstones form the mass list. We do this by selecting the pdg's *)
    goldsnghosts = List[#]&/@Join[Select[Join@@PartList[[All,2]],#[[3]]===U&][[All,9]],Select[Join@@PartList[[All,2]],Last[#]=!=NoGS&][[All,9]]];
    MGMassList = Select[MassList[[2]],Not[MemberQ[goldsnghosts, #[[1]]]]&];
    MGMassList = {MASS, MGMassList};
    MGWidthList = Select[WidthList[[2]],Not[MemberQ[goldsnghosts, #[[1]]]]&];
    MGWidthList = {DECAY, MGWidthList};

(* We have to rename all mass and width symbols that are the same*)
    tempMGdoublemasses = Select[MGMassList[[2]], (Count[MGMassList[[2,All,2]],#[[2]]] !=1)&];
    If[tempMGdoublemasses =!= {}, tmplist = KillDoubles[DeleteCases[tempMGdoublemasses, {_,_,Internal}][[All, 2]]]];
    If[tmplist =!= {}, neweparams = Table[{{dc}, {tmplist[[dc]], NumericalValue[tmplist[[dc]]], False, "Redefined mass"}}, {dc, Length[tmplist]}]];
    tempMGsinglemasses = Complement[MGMassList[[2]], tempMGdoublemasses];
    tempMGdoublemasses=Table[{tempMGdoublemasses[[kk,1]],Symbol[ToString[tempMGdoublemasses[[kk,2]]]<>"MG"<>ToString[kk]],If[tempMGdoublemasses[[kk,3]] === Internal, NumericalValue[tempMGdoublemasses[[kk,2]]], tempMGdoublemasses[[kk,3]]]},{kk,Length[tempMGdoublemasses]}];    
    MGMassList = {MASS, Join[tempMGsinglemasses,tempMGdoublemasses]};
    Do[pdgtomass[MGMassList[[2,kk,1]]] = MGMassList[[2,kk,2]], {kk, Length[MGMassList[[2]]]}];
(* End renaming the masses *)

    Clear[tmplist];


(* we will now do the same for the width *)
    tempMGdoublemasses = Select[MGWidthList[[2]], (Count[MGWidthList[[2,All,2]],#[[2]]] !=1)&];
    If[tempMGdoublemasses =!= {}, tmplist = KillDoubles[DeleteCases[tempMGdoublemasses, {_,_,Internal}][[All, 2]]]];
    If[tmplist =!= {}, neweparams = Join[neweparams, Table[{{dc}, {tmplist[[dc]], NumericalValue[tmplist[[dc]]], False, "Redefined width"}}, {dc, Length[neweparams]+1, Length[neweparams]+Length[tmplist]}]]];
    tempMGsinglemasses = Complement[MGWidthList[[2]], tempMGdoublemasses];
    tempMGdoublemasses=Table[{tempMGdoublemasses[[kk,1]],Symbol[ToString[tempMGdoublemasses[[kk,2]]]<>"MG"<>ToString[kk]],tempMGdoublemasses[[kk,3]]},{kk,Length[tempMGdoublemasses]}];
    MGWidthList = {DECAY, Join[tempMGsinglemasses,tempMGdoublemasses]};
    Do[pdgtowidth[MGWidthList[[2,kk,1]]] = MGWidthList[[2,kk,2]], {kk, Length[MGWidthList[[2]]]}];
(* End renaming the width *)


(* We have to create MGPartList as well, in order to remove the masses of ghosts and goldstones *)   
    MGPartList = Cases[PartList, {{__}, {{__, NoGS}, ___}}];
    MGPartList = ({#1, ReplacePart[ReplacePart[#, pdgtomass[{#[[9]]}], 5], pdgtowidth[{#[[9]]}], 6]&/@ #2}&)@@@MGPartList;
    MGPartList = MGPartList //. {pdgtomass[_] -> ZERO, pdgtowidth[_] -> ZERO};
    GSList = Complement[PartList, MGPartList];
    GSMasses = If[GSList =!= {}, Select[(Sequence @@@ GSList[[All,2]])[[All,5]], Not[MatchQ[#, ZERO]]&], {}];
    GSDecays = If[GSList =!= {}, Select[(Sequence @@@ GSList[[All,2]])[[All,6]], Not[MatchQ[#, ZERO]]&], {}];
    If[Length[MGPartList] != Length[PartList], Print["Warning: Goldstone bosons will be ignored."]];
    MGPartList = ({#1, (Drop[#, -4]&) /@ #2} &)@@@ MGPartList;
    GSList = Cases[MGPartList, {{U[_], _}, {___}}];
    If[GSList =!= {}, GSMasses = Join[GSMasses, Select[(Sequence @@@ GSList[[All,2]])[[All,5]], Not[MatchQ[#, ZERO]]&]]];
    If[GSList =!= {}, GSDecays = Join[GSDecays, Select[(Sequence @@@ GSList[[All,2]])[[All,6]], Not[MatchQ[#, ZERO]]&]]];
    If[Not[FreeQ[MGPartList, U[_]]], MGPartList = DeleteCases[MGPartList, {{U[_], _}, {___}}]; Print["Warning: Ghost fields will be ignored."]];
    parttemplist = Flatten[MGPartList[[All,2]],1];
    MGPartList = MGPartList //. NoPDG -> Identity;
(* End creating MGPartList *)


(* Continue creating MGMassList, MGWidthList, MGEParamList *)
    MixMatList = ToString/@Flatten[MixingMatrix/.PRIVATE`FR$MixingRules [Mix[#]]&/@FR$MassMatrices];
    MGMassList = MGMassList //. NoPDG -> Identity;
    MGMassList = MGMassList //. {a_, b_, Internal} :> {a, b, NumericalValue[b]};
    MGWidthList = MGWidthList //. NoPDG -> Identity;
    MGWidthList = MGWidthList //. {a_, b_, Internal} :> {a, b, NumericalValue[b]};
    Do[If[(MGMassList[[2, ml, 3]] === NoValue[1]) || (MGMassList[[2, ml, 3]] === NoValue), Print["Warning: No mass value found for ", ToString[MGMassList[[2, ml, 2]]], ". Default value 1 assigned."]], {ml, Length[MGMassList]}];
    Do[If[(MGWidthList[[2, ml, 3]] === NoValue[1]) || (MGWidthList[[2, ml, 3]] === NoValue), Print["Warning: No mass value found for ", ToString[MGWidthList[[2, ml, 2]]], ". Default value 1 assigned."]], {ml, Length[MGWidthList]}];
    MGMassList = MGMassList //. NoValue -> Identity;
    MGWidthList = MGWidthList //. NoValue -> Identity;
    MGEParamList = If[neweparams =!= {}, Append[EParamList, {FRMGDefault, neweparams}], EParamList];
    MGEParamList = ({#1, (({#1, DecomposeEParam[#2]}&) @@@ #2)}&) @@@ MGEParamList;
    MGEParamList = ({#1, ({#1, Most[#2]} &) @@@ #2 } &) @@@ MGEParamList;
    Do[If[Head[MGEParamList[[kparam, 1]]] === NoBlockName, Print["Warning: no BlockName found for ", ToString[MGEParamList[[kparam, 2, All, 2, All]][[All, 1]]], ". Default BlockName FRBlock assigned."]],
       {kparam, Length[MGEParamList]}]; 
    MGEParamList = MGEParamList //. NoBlockName -> Identity;
    Do[
      If[Not[FreeQ[ParamList[[kparam]], NoValue]] && Not[Or@@(Not[StringFreeQ[ToString[ParamList[[kparam, 1]]],#]] &/@MixMatList)], 
        Print["Warning: no value found for parameter ", ToString[ParamList[[kparam, 1]]], ". Default value 1 assigned."]],
       {kparam, Length[ParamList]}];
    MGEParamList = MGEParamList //. NoValue -> Identity;

(* Loop through parameters *)
    If[MGEParamList =!= {}, WriteBlock[#,outfile]&/@MGEParamList];
    If[MGMassList =!= {}, WriteBlockMass[MGMassList,outfile]];
    If[MGWidthList =!= {}, WriteBlockDecay[MGWidthList,outfile]];

(* Append QNumbers *)

    WriteQNumbers[outfile];

(* End writing param_card *)
    Close[outfile];
    Print["Output written on ", outfile]
];


(* ::Subsection:: *)
(*ReadLHAFile[]*)


ConvertLHAFile[options___]:=Print["ConvertLHAFile removed.  Please use ReadLHAFile and then WriteParameters instead."];

Options[ReadLHAFile]={Input->"LHAFile.dat",ScreenOutput->True};

ReadLHAFile[options___]:=Block[{infile,lhafile,blocks, params,blockparams,temp,blockelement,lhafilenodecays,
      lhadecays,elemcounter,charcounter,currentline, newblockparams, FoundDecay,line,IsBlockQ,currentblock,MyRule,myfile,
      newparams,newmasses,newwidth, templine, lhafiletemp,ToNewFormat},

(* Input *)
   infile=Input/.{options}/.Options[ReadLHAFile];
   If[Not[FileExistsQ[infile]],Message[Read::NoFile];Abort[]];

(* Preparing the reading: reading the EParamList *)
   blocks=EParamList[[All,1]];
   blocks=ToString/@blocks;
   blocks=Join[blocks,{"MASS"}];
   params=EParamList[[All,2]];
   Do[blockparams[blocks[[kk]]]=params[[kk]],{kk,Length[params]}];
   blockparams["MASS"]=MassList[[2]];
   blockparams["DECAY"]=WidthList[[2]];
   Do[(*Loop over blocks *)
      temp=blocks[[kblock]];
      Do[(* Loop over the elements in this block *)
         blockelement[temp,blockparams[temp][[kelem,1]]]=blockparams[temp][[kelem,2]],
        {kelem,Length[blockparams[temp]]}],
     {kblock,Length[blocks]}];


(* Reading the LHA file *)
   lhafile=Import[infile,"Table"];
   lhafile=DeleteCases[lhafile,{}];
   lhafile=Select[lhafile,((StringTake[ToString[#[[1]]],1] =!= "#"))&];

(*Separate lines *)
   lhafile=If[VectorQ[#]===True,#,Sequence@@#]&/@lhafile;


(*Remove QNumbers *)
   If[Not[FreeQ[lhafile, "QNUMBERS"|"QNumbers"]],
      Print["Omitting QNUMERS."];
      lhafiletemp = lhafile;
      lhafile = {};
      While[lhafiletemp =!= {},
            templine = lhafiletemp[[1]];
            lhafiletemp = Rest[lhafiletemp];
            If[Not[FreeQ[templine, "QNUMBERS"|"QNumbers"]],
               lhafiletemp = Drop[lhafiletemp,4],
               (* else *)
               AppendTo[lhafile, templine]]]];
            

(*Read everything down to DECAY *)
(* A.A. 03.26.2014: Changed following lines in order to allow FR to read LHA File even if the DECAY information
  is put somewhere else than at the end. Previous version of the code is commented below, the new one is below.
   lhafilenodecays={};
   FoundDecay=False;
   line=1;
   While[Not[FoundDecay]&&(line<=Length[lhafile]),
         currentline=lhafile[[line]];
         If[Not[currentline[[1]]==="DECAY"],
            AppendTo[lhafilenodecays,currentline],
            FoundDecay=True];
         line++];
   lhadecays=Select[lhafile,#[[1]]==="DECAY"&];
*)
lhadecays=Cases[lhafile,List["DECAY",__]];
lhafilenodecays=Delete[lhafile,List/@Position[lhafile,"DECAY"][[All,1]]];


(* Read value except decays *)
   lhafile=lhafilenodecays;
   LogOut={};
   While[lhafile=!={},
         currentline=First[lhafile];
         lhafile=Rest[lhafile];
         Which[(currentline[[1]]==="Block")||(currentline[[1]]==="BLOCK"),
               IsBlockQ=MemberQ[blocks,currentline[[2]]];
               If[IsBlockQ,currentblock = currentline[[2]];
                  newblockparams[currentline[[2]]]={};
                  elemcounter=1;
                  AppendTo[LogOut,{}];
                  AppendTo[LogOut,{"Reading Block ", currentline[[2]]}],
                  AppendTo[LogOut,{}];
                  AppendTo[LogOut,{"Block ", currentline[[2]] , " ignored"}]],
               (* else *)
               IsBlockQ,
               AppendTo[LogOut,{"    Reading line ",ToString[elemcounter]}];
               charcounter=1;
               elemcounter++;
               templine={};
               While[(currentline[[charcounter]]=!="#")&&(charcounter<=Length[currentline]),
                     AppendTo[templine,currentline[[charcounter]]];
                     charcounter++];
               AppendTo[newblockparams[currentblock],templine]]];


(* Read decay values *)
   AppendTo[LogOut,{}];
   AppendTo[LogOut,{"Reading Decays"}];
   newblockparams["DECAY"]=Take[Rest[#],2]&/@lhadecays;
(**)

(* Preparing the Write Out *)
(* External parameters *)
   newparams={};
   Do[If[Not[Head[newblockparams[blocks[[kk]]]]===newblockparams],
         newparams=Join[newparams,MyRule[First[blockelement[blocks[[kk]],If[Length[{##}]==2,{First[{##}]},Most[{##}]]]],Last[{##}]]&@@@newblockparams[blocks[[kk]]]]],
     {kk,Length[blocks]-1}];
   newparams=newparams/.ParamRules/.MyRule->Rule;

(* Masses and Wdiths *)
   newmasses=(MyRule[PDGToMass[First[{##}]],Last[{##}]]&@@@newblockparams["MASS"])/.MyRule -> Rule;
   newwidth=(MyRule[PDGToWidth[First[{##}]],Last[{##}]]&@@@newblockparams["DECAY"])/.MyRule -> Rule;

(* Update parameters *)
	UpdateParameters[Flatten[{newparams,newmasses,newwidth}]];

(* Done *)
	Print["Parameters updated from ",infile];
];


